{ ---------------------------------------------------------------------
    $Id: thrBackup.pas,v 1.5 2004/02/12 21:46:16 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup thread - does the actual work.

    See the file COPYING.TXT, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}

unit thrBackup;

interface

{$ifdef win32}
Uses Windows,Classes,Registry,SysUtils,CfgBackup;
{$else}
Uses Libc,Classes,SysUtils,CfgBackup;
{$endif win32}

Type
  TMessageNotifyEvent = Procedure (Msg : String) of Object;

  TBackupThread = Class(TThread)
    FBackupDir : String;
    FBackupExt : String;
    FLogDir : String;
    FDefaultConfig : TBackupConfig;
    FOnInfo,
    FOnError : TMessageNotifyEvent;
    FTotalFiles : Integer;
    FTotalTime : TDateTime;
    FVerbose : Boolean;
  private
    Procedure CompressFile(FN,AFN : String);
    Procedure DoError(Msg : String);overload;
    Procedure DoInfo(Msg : String);overload;
    Procedure DoError(Fmt : String; Args : Array of const);overload;
    Procedure DoInfo(Fmt : String; Args : Array of const);overload;
    procedure InitDefaults;
    Function  BuildBackupFileName(Const Entry,FileName : String; ADate : TDateTime) : String;
    Procedure PerformBackup(Entry : String; Const Config : TBackupConfig);
    Function  InitBackupData(const Entry: String; Var Config : TBackupConfig) : Boolean;
    function FillPlaceholders(const Entry, FileName: String; ADate: TDateTime): String;
    function BuildLogFileName(const Entry, FileName: String; ADate: TDateTime): String;
    function SetupLog(FN: String): TStream;
    procedure CompressBackupFile(FN: String);
    function SkipBackup(C: TBackupConfig): boolean;
  Public
    Procedure Execute; override;
    Property TotalFiles : Integer Read FTotalFiles;
    Property TotalTime : TDateTime Read FTotalTime;
    Property Verbose : Boolean Read FVerbose Write FVerbose;
    Property OnInfoMessage : TMessageNotifyEvent Read FOnInfo Write FOnInfo;
    Property OnErrorMessage : TMessageNotifyEvent Read FOnError Write FOnError;
  end;


ResourceString
  SErrNoData          = 'No backup data found for %s';
  SErrTerminated      = 'Scheduled backup terminated. Cleared %d files in a total time of %s';
  SErrUnexpected      = 'An unexpected error occurred while backing up databases: %s';
  SErrUnexpectedEntry = 'An unexpected error occurred while backing up entry "%s": %s';
  SProcessing         = 'Processing %d backup entries.';
  SBackingup          = 'Backing up entry "%s": Database "%s" on server "%s"';
  SUsingBackupFile    = '  Using backup file: %s';
  SUsingLogFile       = '  Using log file: %s';
  SNoLogFile          = '  No log file';
  SSkippingBackup        = 'Skipping back up entry "%s"';
  SFileNameEqualsArchive = 'Backup file name equals compression file name: "%s"';
  SErrCompressing        = 'An error occurred when compressing "%s" to "%s": "%s"';

implementation

{$ifdef useabbrevia}
uses IBServices,abzipper,abarctyp;

Const
   ArchiveExt = '.zip';

{$else}
uses IBServices,ZLib;

Const
  BufSize = 4096;
  ArchiveExt = '.gz';
{$endif}

{$IFNDEF LINUX}
const
  PathDelim='\';
{$ENDIF}

{ TBackupThread }

{ ---------------------------------------------------------------------
  Initialization and configuration, auxiliary routines
  ---------------------------------------------------------------------}

procedure TBackupThread.InitDefaults;

Var
  Conf : TBackupDefaults;

begin
  if ReadDefaults(Conf) then
    begin
    FBackupDir:=Conf.BackupDir;
    FLogDir:=Conf.LogDir;
    FBackupExt:=Conf.BackupExt;
    end;
  ReadConfig(SKeyDefaults,FDefaultConfig);
end;


Function TBackupThread.InitBackupData(Const Entry : String; var Config : TBackupConfig) : Boolean;

begin
  Config:=FDefaultConfig;
  Result:=ReadConfig(Entry,Config);
  If Result then
    With Config do
      Result:=(DatabasePath<>'') and
              (UserName<>'');
end;


function TBackupThread.FillPlaceholders(Const Entry,FileName : String; ADate : TDateTime) : String;

Var
  S : String;
  N : Integer;

begin
  Result:=StringReplace(FileName,'%D',FormatDateTime('yyyy-mm-dd',ADate),[rfReplaceAll]);
  Result:=StringReplace(Result,'%T',FormatDateTime('hh-nn-ss',ADate),[rfReplaceAll]);
  S:=StringReplace(Entry,':','_',[rfReplaceAll]);
  S:=StringReplace(S,'\','_',[rfReplaceAll]);
  S:=StringReplace(S,'/','_',[rfReplaceAll]);
  Result:=StringReplace(Result,'%E',S,[rfReplaceAll]);
  // For remote hosts, it will always be N=1, but this can't be helped.
  If Pos('%N',Result)<>0 then
    begin
    N:=0;
    Repeat
      Inc(N);
      S:=StringReplace(Result,'%N',IntToStr(N),[rfReplaceAll]);
    Until Not FileExists(S);
    Result:=S;
    end;
end;


function TBackupThread.BuildLogFileName(Const Entry,FileName : String; ADate : TDateTime): String;

Var
  S : String;

begin
  Result:='';
  If (FileName<>'') then
    begin
    If (Pos(PathDelim,FileName)<>0) then
      S:=FileName
    else
      {$IFNDEF LINUX}
      S:=IncludeTrailingBackslash(FLogDir)+'%E.log';
      {$ELSE}
      S:=IncludeTrailingPathDelimiter(FLogDir)+'%E.log';
      {$ENDIF}
    Result:=FillPlaceHolders(Entry,S,Adate);
    end;
end;


function TBackupThread.BuildBackupFileName(Const Entry,FileName : String; ADate : TDateTime): String;

Var
  S : String;

begin
  If (FileName='') then
    {$IFNDEF LINUX}
    S:=IncludeTrailingBackSlash(FBackupDir)+'%E'+FBackupExt
    {$ELSE}
    S:=IncludeTrailingPathDelimiter(FBackupDir)+'%E'+FBackupExt
    {$ENDIF}
  else
    begin
    //!! This is not good on a remote host...
    If (Pos(PathDelim,FileName)<>0) then
      S:=FileName
    else
      {$IFNDEF LINUX}
      S:=IncludeTrailingBackslash(FBackupDir)+FileName;
      {$ELSE}
      S:=IncludeTrailingPathDelimiter(FBackupDir)+FileName;
      {$ENDIF}
    If (ExtractFileExt(S)='') then
      S:=ChangeFileExt(S,FBackupExt);
    end;
  Result:=StringReplace(FillPlaceHolders(Entry,S,Adate),' ','_',[rfReplaceAll]);
end;

Function TBackupThread.SetupLog(FN : String) : TStream;

begin
  Result:=Nil;
  If (FN<>'') then
    Try
      Result:=TFileStream.Create(FN,fmCreate);
    except
      Result:=Nil;
    end;
end;

Function TBackupThread.SkipBackup(C : TBackupConfig) : boolean;

Var
  Dow : Integer;

begin
  Result:=C.Skip;
  If (Not result) and (C.DaysOfWeek<>0) then
    begin
    DOW:=1 shl (DayOfWeek(Date)-1); // Bit 0 is sunday, bit 6 is saturday.
    Result:=(C.DaysOfWeek and DOW)=0;
    end;
end;


{ ---------------------------------------------------------------------
  Reporting routines.
  ---------------------------------------------------------------------}

procedure TBackupThread.DoError(Fmt: String; Args: array of const);
begin
  DoError(Format(Fmt,Args));
end;


procedure TBackupThread.DoInfo(Fmt: String; Args: array of const);
begin
  DoInfo(Format(Fmt,Args));
end;


procedure TBackupThread.DoError(Msg: String);
begin
  If Assigned(FOnError) then
    FOnError(Msg);
end;


procedure TBackupThread.DoInfo(Msg: String);
begin
  If FVerbose and Assigned(FOnInfo) then
    FOnInfo(Msg);
end;


{ ---------------------------------------------------------------------
  Routines that do the real work.
  ---------------------------------------------------------------------}


procedure TBackupThread.Execute;

Var
  I : Integer;
  Config : TBackupConfig;
  L : TStrings;

begin
  Try
    InitDefaults;
    L:=TStringList.Create;
    Try
      EnumEntries(L);
      DoInfo(Format(SProcessing,[L.Count]));
      I:=0;
      FTotalFiles:=0;
      FTotalTime:=Now;
      While (Not Terminated) and (I<L.Count) do
        begin
        Config:=FDefaultConfig;
        If InitBackupData(L[i],Config) then
          begin
          If SkipBackup(Config) then
            DoInfo(SSKippingBackup,[L[i]])
          else
            PerFormBackup(L[i],Config)
          end
        else
          DoError(Format(SErrNoData,[L[I]]));
        Inc(I);
        end;
      FTotalTime:=Now-FTotalTime;
      If Terminated then
        DoError(Format(SErrTerminated,[FTotalFiles,TimeToStr(FtotalTime)]));
    Finally
      L.Free;
    end;
  except
    On E : Exception do
      DoError(Format(SErrUnexpected,[E.Message]));
  end;
end;


{ ----------------------------------------------------------------------
  Backup implementation using IBX.
  ----------------------------------------------------------------------}

procedure TBackupThread.PerformBackup(Entry: String;Const  Config: TBackupConfig);

  Function LocalBackup (BS : TIBBackupService) : Boolean;

  begin
    With BS do
      Result:=(Protocol=Local) or
              ((Protocol=TCP) and (CompareText(ServerName,'localhost')=0));
  end;

Var
  LogStream               : TStream;
  LogLine,LogFileName     : String;
  I                       : Integer;
  IB : TIBBackupService;
begin
  Try
    IB:=TIBBackupService.Create(Nil);
    With IB do
      Try
        LoginPrompt:=False;
        ServerName:=Config.ServerName;
        If (ServerName='') then
          begin
          If LocalIsSafe then
            Protocol:=Local
          else
            begin
            ServerName:='localhost';
            Protocol:=TCP
            end;
          end
        else
          Protocol:=TCP;
        Params.Values['user_name']:=Config.UserName;
        Params.Values['password']:=Config.PassWord;
        DatabaseName:=Config.DatabasePath;
        BackupFile.CommaText:=BuildBackupFileName(Entry,Config.BackupFile,Now);
        LogFileName:=BuildLogFileName(Entry,Config.LogFileName,Now);
        LogStream:=SetUpLog(LogFileName);
        DoInfo(SBackingup,[Entry,DatabaseName,ServerName]);
        For I:=0 to BackupFile.Count-1 do
          DoInfo(SUsingBackupFile,[BackupFile[i]]);
        If (LogStream<>Nil) then
          DoInfo(SUsingLogFile,[LogFileName])
        else
          DoInfo(SNoLogFile);
        Try
          Verbose:=(LogStream<>Nil);
          Active:=True;
          Try
            ServiceStart;
            If Verbose then
              begin
              While Not eof do
                begin
                LogLine:=GetNextLine+#13#10;
                LogStream.Write(LogLine[1],Length(LogLine));
                end
              end
            else
              While IsServiceRunning do
                Sleep(10);
          Finally
            Active:=False;
          end;
          If Config.CompressBackup and LocalBackup(IB) then
              For I:=0 to BackupFile.Count-1 do
                If FileExists(BackupFile[i]) then
                  CompressBackupFile(BackupFile[i]);
          Inc(FTotalFiles);
        Finally
          FreeAndNil(LogStream);
        end;
      Finally
        IB.Free;
      end;
  except
    On E : Exception do
      DoError(Format(SErrUnexpectedEntry,[Entry,E.Message]));
  end;
end;

// This function should not raise exceptions.

Procedure TBackupThread.CompressBackupFile(FN : String);

Var
  ZFN : String;

begin
  Try
    ZFN:=ChangeFileExt(FN,ArchiveExt);
    If (ZFN=FN) then
      Raise Exception.CreateFmt(SFileNameEqualsArchive,[FN]);
    CompressFile(FN,ZFN);
    // If we are here, compress was OK, delete original.
    If Not DeleteFile(FN) then
      {$IFDEF WIN32}
      RaiseLastWin32Error;
      {$ELSE}
      RaiseLastOSError;
      {$ENDIF}
 except
   On E : Exception do
    DoError(SErrCompressing,[FN,ZFN,E.Message]);
  end;
end;

{$ifdef UseAbbrevia}
Procedure TBackupThread.CompressFile(FN,AFN : String);

begin
  With TABZipper.Create(Nil) do
    Try
      StoreOptions:=[soStripPath];
      AutoSave:=true;
      FileName:=AFN;
      AddFiles(FN,0);
    Finally
      Free;
    end;
end;

{$else}
Procedure TBackupThread.CompressFile(FN,AFN : String);

Var
  InF,OutF : TFileStream;
  CS : TCompressionStream;
  Buf : Pointer;
  Count : Integer;

begin
  Inf:=TFileStream.Create(FN,fmOpenRead);
  Try
    OutF:=TFileStream.Create(AFN,fmCreate);
    Try
      GetMem(Buf,BufSize);
      Try
        CS:=TCompressionStream.Create(clMax,OutF);
        Try
          Repeat
            Count:=Inf.Read(Buf^,BufSize);
            If (Count>0) then
              CS.Write(Buf^,Count);
          Until (Count<BufSize);
        Finally
          CS.Free;
        end;
      Finally
        FreeMem(Buf);
      end;
    Finally
      OutF.Free;
    end;
  Finally
    Inf.Free;
  end;
end;
{$endif}


end.

