{ ---------------------------------------------------------------------
    $Id: ctrlBackup.pas,v 1.1 2003/10/12 10:56:43 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup service control routines for managing programs
    
    See the file COPYING.TXT, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}
unit ctrlBackup;

interface

uses CfgBackup,SysUtils,Classes;

{ ---------------------------------------------------------------------
  Backup service control functions.
  ---------------------------------------------------------------------}

Function GetBackupServiceStatus : TBackupServiceStatus;

// These can raise an exception.
Procedure StartBackupService;
Procedure StopBackupService;
Procedure PauseBackupService;
Procedure ContinueBackupService;
Procedure ReloadBackupService;

implementation

{$ifdef linux}

{ ---------------------------------------------------------------------
  Linux implementation of service control functions.
  ---------------------------------------------------------------------}

uses Libc;

Resourcestring
  SErrNoPIDFound     = 'Could not determine Process ID of backup service.';
  SErrNotImplemented = 'This feature is not yet implemented';

Type
  EServiceManager = Class(Exception);

Procedure DoError(Const Msg : String);

begin
  Raise EServiceManager.Create(Msg);
end;

Function GetServicePID : Integer;

Var
  FN,S : String;

begin
  Result:=-1;
  FN:=GetPIDFileName;
  If FileExists(FN) then
    begin
    With TFileStream.Create(FN,fmOpenRead) do
      try
        SetLength(S,Size);
        Read(S[1],Size);
      finally
        Free;
      end;
    Result:=StrToIntDef(S,-1);
    end;
end;

Function GetPIDStatus(PID : Integer) : String;

Var
  FN,S : String;
  F : Text;
  B : Boolean;

begin
  Result:='';
  FN:=Format('/proc/%d/status',[PID]);
  If FileExists(FN) then
    begin
    AssignFile(F,FN);
    Reset(F);
    Try
      Repeat
        Readln(F,S);
        B:=(Pos('State:',S)<>0);
      Until EOF(F) or B;
      If B then
        begin
        Delete(S,1,7);
        S:=Trim(S);
        Result:=S;
        end;
    finally
      Close(F);
    end;
    end;
end;

Function GetBackupServiceStatus : TBackupServiceStatus;

Var
  S : String;
  PID : Integer;

begin
  Result:=bssUnknown;
  Try
    PID:=GetServicePID;
    If (PID<>-1) then
      begin
      S:=GetPIDStatus(PID);
      If (Length(S)>0) then
        begin
        Result:=bssRunning;
        If (S[1]='T') then
          Result:=bssPaused;
        end;
      end;
  except
    // Nothing.
  end;
end;

Procedure StartBackupService;

begin
  DoError(SErrNotImplemented);
end;

Procedure KillService(Sig : Integer);

Var
  PID : Integer;

begin
  PID:=GetServicePID;
  If (PID=-1) then
    DoError(SErrNoPIDFound);
  If Kill(PID,Sig)<>0 then
    RaiseLastOSError;
end;

Procedure StopBackupService;

begin
  KillService(SIGTERM);
end;

Procedure PauseBackupService;

begin
  KillService(SIGSTOP);
end;

Procedure ContinueBackupService;

begin
  KillService(SIGCONT);
end;

Procedure ReloadBackupService;

begin
  KillService(SIGHUP);
end;
{$else}

{ ---------------------------------------------------------------------
  Windows implementation of service functions.
  ---------------------------------------------------------------------}
uses servicemanager,winsvc;


Function GetBackupServiceStatus : TBackupServiceStatus;

Var
  FLastStatus : Cardinal;
  STatus : TServiceStatus;


begin
  Result:=bssUnknown;
  Try
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      GetServiceStatus(SBackupService,Status);
      FLastStatus:=Status.dwCurrentState;
      if FLAstStatus=SERVICE_RUNNING then
        Result:=bssRunning
      else if FLastStatus=SERVICE_STOPPED then
        Result:=bssStopped
      else If FLastStatus=SERVICE_PAUSED Then
        result:=bssPaused;
      Disconnect;
    Finally
      Free;
    end;
  except
  end;
end;

Procedure StartBackupService;

begin
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      StartService(SBackupService,Nil);
      Disconnect;
    Finally
      Free;
    end;
end;

Procedure StopBackupService;

begin
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      StopService(SBackupService,False);
      Disconnect;
    Finally
      Free;
    end;
end;

Procedure PauseBackupService;

begin
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      PauseService(SBackupService);
      Disconnect;
    Finally
      Free;
    end;
end;

Procedure ContinueBackupService;

begin
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      ContinueService(SBackupService);
      Disconnect;
    Finally
      Free;
    end;
end;

Procedure ReloadBackupService;

begin
  With TServiceManager.Create(Nil) do
    Try
      Connect;
      CustomControlService (SBackupService,ConfigControlCode);
      Disconnect;
    Finally
      Free;
    end;
end;

{$endif}

end.
