{ ---------------------------------------------------------------------
    $Id: CfgBackup.pas,v 1.5 2003/12/23 18:41:16 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup service configuration routines.

    See the file COPYING.TXT, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}
unit CfgBackup;

interface

{$ifdef win32}
uses Windows,Registry,TypInfo,Sysutils,Classes;
{$else}
uses LibC,IniFiles,TypInfo,Sysutils,Classes;
{$endif}

{ ---------------------------------------------------------------------
  Global constants used in configuration stuff.
  ---------------------------------------------------------------------}

Const
{$ifdef win32}
  SKeyBackup       = '\Software\FIBSS\fibss\';
  SKeySchedule     = SKeyBackup+'Schedule';
  SKeyDefaults     = SKeyBackup+'Defaults';
  SKeyBackups      = SkeyBackup+'Backups\';
{$else}
  SKeyBackup       = 'Default';
  SKeySchedule     = 'Schedule';
  SKeyDefaults     = 'Defaults';
  SKeyBackups      = '';
  GlobalIniFile    = '/etc/fibss.cfg';
  LocalIniFile     = '.fibss.cfg';
  GlobalPIDFile    = '/var/run/fibss.pid';
  LocalPIDFile     = '.fibss.pid';
{$endif}
  SKeyBackupExt    = 'BackupExtension';
  SKeyBackupDir    = 'BackupDir';
  SKeyLogDir       = 'Logdir';
  SKeyLogFile      = 'LogFile';
  SKeyTime         = 'Time';
  SKeyServerName   = 'ServerName';
  SKeyDatabasePath = 'Path';
  SKeyBackupFile   = 'BackupFile';
  SKeyUserName     = 'UserName';
  SKeyPassWord     = 'Password';
  SKeyOptions      = 'Options';
  SKeyCompress     = 'Compress';
  SKeySkip         = 'Skip';
  SKeyDaysOfWeek   = 'DaysOfWeek';

  SBackupService   = 'FIBService';
  SDefault         = '(default)';
  DefaultExtension = '.gbk';

  ConfigControlCode  = 129;

{ ---------------------------------------------------------------------
  Special types
  ---------------------------------------------------------------------}

Type
  TBackupServiceStatus = (bssRunning,bssNotInstalled,bssPaused,bssStopped,bssReload,bssUnknown);

  TBackupOption = (boIgnoreChecksums, boIgnoreLimbo, boMetadataOnly,
                   boNoGarbageCollection, boNonTransportable,
                   boConvertExtTables);

  TBackupOptions = Set of TBackupOption;

  TBackupDefaults = Record
    BackupDir,
    LogDir,
    BackupExt : String;
    BackupTime : TDateTime;
  end;

  TBackupConfig = Record
    ServerName   : String;
    DatabasePath : String;
    BackupFile   : String;
    UserName     : String;
    Password     : String;
    Options      : TBackupOptions;
    LogFileName  : String;
    DaysOfWeek   : Byte;      // 0 = All days, otherwise day mask:
                              // Bit 0 is sunday, bit 6 is saturday.
    Skip         : Boolean;   // Not yet used.
    CompressBackup : Boolean; // Not yet used. Only local.
  end;

const
  BackupServiceStatusNames:array[TBackupServiceStatus] of String=('RUNNING','NOTINSTALLED','PAUSED','STOPPED','RELOAD','UNKNOWN');
{ ---------------------------------------------------------------------
  Global Auxiliary functions
  ---------------------------------------------------------------------}

Function StringToBackupOptions (Const S : String) : TBackupOptions;
Function BackupOptionsToString (Options : TBackupOPtions) : String;
Function GetStatusDescription (S : TBackupServiceStatus) : String;

{ ---------------------------------------------------------------------
  Actual configuration functions.
  ---------------------------------------------------------------------}

Function ReadConfig(Entry : String; Var Config : TBackupConfig) : Boolean; overload;
Function WriteConfig(Entry : String; Const Config : TBackupConfig) : Boolean; overload;
Function ReadDefaults(Var Config : TBackupDefaults) : boolean;
Function WriteDefaults(Var Config : TBackupDefaults) : Boolean;
Function EnumEntries(List : TStrings) : Boolean;
Function DeleteEntry(Entry : String) : Boolean;
Function LocalIsSafe : Boolean; // Safe to use local protocol ?

{$ifdef linux}
Function GetIniFileName : string;
Function GetPIDFileName : string;
{$endif}

Resourcestring
  SRunning      = 'Running';
  SNotInstalled = 'Not installed.';
  SPaused       = 'Paused';
  SStopped      = 'Not running';
  SReload       = 'Reloading';
  SUnknown      = 'Unknown';

implementation

{$ifdef linux}

{ ---------------------------------------------------------------------
  Linux auxiliary functions.
  ---------------------------------------------------------------------}

var
  CmdLineIniFile  : String = '';
  CmdLinePIDFile  : String = '';

procedure ProcessCommandLineOptions ;

  Function NextParam (Var Index : Integer) : String;

  begin
    If (Index<ParamCount) then
      begin
      Inc(Index);
      Result:=Paramstr(Index);
      end
    Else
      Result:='';
  end;

Var
  I : integer;

begin
  CmdLineIniFile:='';
  CmdLinePIDFile:='';
  I:=0;
  While I<ParamCount do
    begin
    Inc(I);
    If (Paramstr(i)='-c') then
      CmdLineIniFile:=NextParam(i)
    else If Paramstr(i)='-p' then
      CmdLinePIDFile:=NextParam(i);
    end;
end;


Function GetHomeDir : String;

begin
  Result:=IncludeTrailingPathDelimiter(GetEnvironmentVariable('HOME'));
end;

Function GetInifileName : string;

begin
  If (CmdLineIniFile<>'') then
    Result:=CmdLineIniFile
  else If (GetUID=0) then
    Result:=GlobalInifile
  else
    Result:=GetHomeDir+LocalInifile;
end;

Function GetPIDFileName : string;

begin
  If (CmdLinePIDFile<>'') then
    Result:=CmdLinePIDFile
  else If (GetUID=0) then
    Result:=GlobalPIDfile
  else
    Result:=GetHomeDir+LocalPIDfile;
end;

Function DefaultBackupDir : String;

begin
  Result:='/tmp/';
end;

Function EntryToKey(Entry : String) : String;

begin
  If CompareText(Entry,SDefault)=0 then
    Result:=SKeyDefaults
  else
    Result:=Entry;
end;

{ ---------------------------------------------------------------------
  TConfigObject Mimics Tregistry on Linux.
  ---------------------------------------------------------------------}

Type
  // Mimic TRegistry
  TConfigObject = Class(Tinifile)
    FCurrentKey : String;
    FReadOnly : Boolean;
  Public
    Constructor Create(ReadOnly : Boolean);
    Destructor destroy; override;
    Procedure EnumSubKeys(KeyName : String; List : TStrings);
    Function OpenKeyReadOnly(Const Key : String) : Boolean;
    Function OpenKey(Const Key : String; AllowCreate : Boolean) : Boolean;
    Function ReadString(Const Name : String) : String;
    Procedure WriteString(Const Name,Value : String);
    Function ReadTime(Const Name : String) : TDateTime;
    Procedure WriteTime(Const Name: String; Value : TDateTime);
    Function ReadBool(Const Name : String) : Boolean;
    Procedure WriteBool(Const Name: String; Value : Boolean);
    Function ReadInteger(Const Name : String) : Integer;
    Procedure WriteInteger(Const Name: String; Value : Integer);
    Function ValueExists(Name : String) : Boolean;
    Procedure DeleteValue(Name : String);
    Procedure DeleteKey(Name : String);
    Procedure CloseKey;
  end;

Constructor TConfigObject.Create(ReadOnly : Boolean);

begin
  Inherited create(getinifilename);
  FReadOnly:=ReadOnly;
  FCurrentKey:=SKeyBackup;
end;

Destructor TConfigObject.destroy;

begin
  If Not FReadOnly then
    UpdateFile;
  Inherited;
end;

Function TConfigObject.OpenKeyReadOnly(Const Key : String) : Boolean;

begin
  FCurrentkey:=StringReplace(Key,'\','_',[rfReplaceAll]);
  Result:=True;
end;

Function TConfigObject.OpenKey(Const Key : String; AllowCreate : Boolean) : Boolean;

begin
  Result:=AllowCreate or SectionExists(Key);
  If Result then
    FCurrentKey:=Key;
end;

Procedure TConfigObject.CloseKey;
begin
  If Not FReadOnly then
    UpdateFile;
end;

Procedure TConfigObject.EnumSubKeys(KeyName : String; List : TStrings);

Var
  I : Integer;

begin
  ReadSections(List);
  I:=List.IndexOf(SkeySchedule);
  If I<>-1 then
    List.Delete(i);
  I:=List.IndexOf(SkeyDefaults);
  If I<>-1 then
    List.Delete(I);
end;

Function TConfigObject.ValueExists(Name : String) : Boolean;

begin
  Result:=Inherited ValueExists(FCurrentkey,Name);
end;

Procedure TConfigObject.DeleteValue(Name : String) ;

begin
  Inherited DeleteKey(FCurrentkey,Name);
end;

Procedure TConfigObject.DeleteKey(Name : String);

begin
  Inherited EraseSection(Name);
end;


Function TConfigObject.ReadString(Const Name : String) : String;

begin
  Result:=Inherited ReadString(FCurrentKey,Name,'');
end;

Procedure TConfigObject.WriteString(Const Name,Value : String);

begin
   Inherited WriteString(FCurrentKey,Name,Value);
end;

Function TConfigObject.ReadTime(Const Name : String) : TDateTime;

begin
  Result:=Inherited ReadTime(FCurrentKey,Name,0);
end;

Procedure TConfigObject.WriteTime(Const Name: String; Value : TDateTime);

begin
  Inherited WriteTime(FCurrentKey,Name,Value);
end;

Function TConfigObject.ReadBool(Const Name : String) : Boolean;

begin
  Result:=Inherited ReadBool(FCurrentKey,Name,False);
end;

Procedure TConfigObject.WriteBool(Const Name: String; Value : Boolean);

begin
  Inherited WriteBool(FCurrentKey,Name,Value);
end;

Function TConfigObject.ReadInteger(Const Name : String) : Integer;

begin
  Result:=Inherited ReadInteger(FCurrentKey,Name,0);
end;

Procedure TConfigObject.WriteInteger(Const Name: String; Value : Integer);

begin
  Inherited WriteInteger(FCurrentKey,Name,Value);
end;

{$else}

{ ---------------------------------------------------------------------
  Windows Auxiliary routines
  ---------------------------------------------------------------------}

Var
  UseCurrentUser : Boolean = False;

procedure ProcessCommandLineOptions ;

begin
  UseCurrentUser:=FindCmdLineSwitch('c',['/','-'],False)
end;

Function DefaultBackupDir : String;

begin
  Result:=ExtractFilePath(ParamStr(0)); // FallBack
end;

{ ---------------------------------------------------------------------
  Windows TConfigObject is a simple TRegistry descendent
  ---------------------------------------------------------------------}

Type
  TConfigObject = Class(TRegistry)
  Public
    Constructor Create(ReadOnly : Boolean);
    Procedure EnumSubKeys(KeyName : String; List : TStrings);
  end;

Function EntryToKey(Entry : String) : String;

begin
  If Entry=SDefault then
    Result:=SKeyDefaults
  else
    Result:=SKeyBackups+Entry;
end;

{ TConfigObject }

constructor TConfigObject.Create(ReadOnly: Boolean);
begin
  Inherited Create;
  If UseCurrentUser then
    RootKey:=HKEY_CURRENT_USER
  else
    RootKey:=HKEY_LOCAL_MACHINE;
  If ReadOnly then
    Access:=KEY_READ;
end;

procedure TConfigObject.EnumSubKeys(KeyName: String; List: TStrings);
begin
  OpenKeyReadOnly(KeyName);
  GetKeyNames(List);
end;

{$endif}

{ ---------------------------------------------------------------------
  Global configuration routines - cross-platform
  ---------------------------------------------------------------------}

Function ReadDefaults(Var Config : TBackupDefaults) : Boolean;

begin
  Result:=False;
  With TConfigObject.Create(True),Config do
    Try
    If OpenKeyReadOnly(SKeyDefaults) then
      begin
      If ValueExists(SKeyBackupDir) then
        BackupDir:=ReadString(SKeyBackupDir)
      else
        BackupDir :=DefaultBackupDir;
      If ValueExists(SKeyLogDir) then
        LogDir:=ReadString(SKeyLogDir)
      else
        LogDir:=Config.BackupDir;
      If ValueExists(SKeyBackupExt) then
        BackupExt:=ReadString(SKeyBackupExt)
      else
        BackupExt:=DefaultExtension; // FallBack
      Result:=True;
      end;
    If OpenKeyReadOnly(SKeySchedule) then
      begin
      If ValueExists(SKeyTime) then
        BackupTime:=ReadTime(SKeyTime)
      else
        BackupTime:=EncodeTime(0,0,0,0);
      end;
    Finally
      Free;
    end;
end;

Function WriteDefaults(Var Config : TBackupDefaults) : Boolean;

Var
  C : TConfigObject;

  Procedure UpdateValue(Const KeyName,KeyValue : String);

  begin
    With C do
      begin
      If (KeyValue<>'') then
        C.WriteString(KeyName,KeyValue)
      else if C.ValueExists(KeyName) then
        C.DeleteValue(KeyName);
      end
  end;

begin
  Result:=False;
  c:=TConfigObject.Create(False);
  With C,Config do
    Try
    If OpenKey(SKeyDefaults,True) then
      begin
      UpdateValue(SKeyBackupDir,BackupDir);
      UpdateValue(SKeyLogDir,LogDir);
      UpdateValue(SKeyBackupExt,BackupExt);
      Result:=True;
      end;
    If OpenKey(SKeySchedule,True) then
      WriteTime(SKeyTime,BackupTime);
    Finally
      Free;
    end;
end;

Function ReadConfig(Entry : String; Var Config : TBackupConfig) : Boolean;

begin
  With TConfigObject.Create(True) do
    begin
    Result:=OpenKeyReadOnly(EntryToKey(Entry));
    If result then
      try
        With Config do
          begin
          If ValueExists(SKeyServerName) then
            ServerName:=ReadString(SKeyServerName);
          If ValueExists(SKeyDatabasePath) then
            databasePath:=ReadString(SKeyDatabasePath);
          If ValueExists(SKeyBackupFile) then
            BackupFile:=ReadString(SKeyBackupFile);
          If ValueExists(SKeyLogFile) then
            LogFileName:=ReadString(SKeyLogFile);
          If ValueExists(SKeyUserName) then
            UserName:=ReadString(SKeyUserName);
          If ValueExists(SKeyPassword) then
            Password:=ReadString(SKeyPassWord);
          If ValueExists(SKeyOptions) then
            Options:=StringToBackupOptions(ReadString(SKeyOptions));
          If ValueExists(SKeyCompress) then
            CompressBackup:=ReadBool(SKeyCompress)
          else
            CompressBackup:=False;
          If ValueExists(SKeySkip) then
            Skip:=ReadBool(SKeySkip)
          else
            Skip:=False;
          If ValueExists(SkeyDaysOfWeek) then
            DaysOfWeek:=ReadInteger(SkeyDaysOfWeek)
          else
            DaysOfWeek:=0;
          end;
      finally
        CloseKey;
      end;
    end;
end;


Function WriteConfig(Entry : String; Const Config : TBackupConfig) : Boolean; overload;

Var
  C : TConfigObject;

  Procedure UpdateValue(Const KeyName,Value: String);

  begin
    If (Value<>'') then
      C.WriteString(KeyName,Value)
    else if C.ValueExists(KeyName) then
      C.DeleteValue(KeyName)
  end;

begin
  C:=TConfigObject.Create(False);
  With c do
    Try
      Result:=OpenKey(EntryToKey(Entry),True);
      If Result then
        try
          With Config do
            begin
            UpdateValue(SKeyServerName,ServerName);
            UpdateValue(SKeyDatabasePath,DatabasePath);
            UpdateValue(SKeyBackupFile,BackupFile);
            UpdateValue(SKeyLogFile,LogFileName);
            UpdateValue(SKeyUserName,UserName);
            UpdateValue(SKeyPassword,Password);
            UpdateValue(SKeyOptions,BackupOptionsToString(Options));
            WriteBool(SKeyCompress,CompressBackup);
            WriteBool(SkeySkip,Skip);
            WriteInteger(SKeyDaysOfWeek,DaysOfWeek);
            end;
        finally
          CloseKey;
        end;
    Finally
      Free;
    end;
end;

Function EnumEntries(List : TStrings) : Boolean;

begin
  Result:=True;
  With TConfigObject.Create(True) do
    Try
      EnumSubKeys(SKeyBackups,List);
    Finally
      Free;
    end;
end;

Function DeleteEntry(Entry : String) : Boolean;

begin
  Result:=True;
  With TConfigObject.Create(False) do
    Try
      DeleteKey(EntryToKey(Entry));
    Finally
      Free;
    end;
end;

Function LocalIsSafe : Boolean;
{
   Safe to use local protocol ?

   On NT, this is only so if both the scheduling service and the IBServer
   service are running with the 'Interact With Desktop' flag set.
   Otherwise, the local protocol will not work.
   TCP/IP should be used instead.
   Later on, a detection mechanism should be installed.
   (use service manager to detect flag of both services)

   On Linux it should be safe, but the second backup
   always fails with 'Cannot connect to services manager'
   when using the local protocol.

   So for the time being:
   Local is NOT safe under any circumstances...
}
begin
  Result:=False;
end;


{ ---------------------------------------------------------------------
  Global cross-platform auxialiary routines.
  ---------------------------------------------------------------------}

Var
  OptionNames : Array[TBackupOption] of string;

Function StringToBackupOptions (Const S : String) : TBackupOptions;

Var
  CO,OS : String;
  I : Integer;
  O : TBackupOption;

begin
  Result:=[];
  OS:=S;
  While (OS<>'') do
    begin
    I:=Pos(',',OS);
    If (I=0) then
      I:=Length(OS)+1;
    CO:=Copy(OS,1,I-1);
    Delete(OS,1,I);
    For O:=Low(TbackupOption) to High(TBackupOption) do
      if CompareText(OptionNames[O],CO)=0 then
        Include(Result,O);
    end;
end;

Function BackupOptionsToString (Options : TBackupOPtions) : String;

Var
  O : TbackupOption;

begin
  Result:='';
  For O:=Low(TBackupOption) to High(TBackupOption) do
    If O in options then
      If (Result='') then
        Result:=OptionNames[O]
      else
        Result:=Result+','+OptionNames[O];
end;

Procedure InitOptionNames;

Var
  O : TbackupOption;
  S : String;

begin
  For O:=Low(TBackupOption) to High(TBackupOption) do
    begin
    S:=GetEnumName(TypeInfo(TBackupOption),Ord(O));
    Delete(S,1,2);
    OptionNames[O]:=S;
    end;
end;


Function GetStatusDescription (S : TBackupServiceStatus) : String;

begin
  Case S of
    bssRunning      : Result:=SRunning;
    bssNotInstalled : Result:=SNotInstalled;
    bssPaused       : Result:=SPaused;
    bssStopped      : Result:=SStopped;
    bssReload       : Result:=SReload;
    bssUnknown      : Result:=SUnknown;
  end;
end;

{ ---------------------------------------------------------------------
  Initialization
  ---------------------------------------------------------------------}


Initialization
  InitOptionNames;
  ProcessCommandLineOptions;
end.
