{ ---------------------------------------------------------------------
    $Id: frmMain.pas,v 1.5 2004/02/12 21:55:25 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup service manager main screen
    
    See the file COPYING.TXT, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}
unit frmMain;

interface

uses
  {$IFDEF WIN32}
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, Mask, ExtCtrls, WinSVC, ServiceManager, ActnList,
  Grids,cfgBackup, Buttons, IdBaseComponent, IdComponent, IdTCPConnection,
  IdTCPClient;
  {$ELSE}
  SysUtils, Classes, QGraphics, QControls, QForms,
  QDialogs, QStdCtrls, QMask, QExtCtrls, QActnList,
  QGrids,CfgBackup, QButtons, IdBaseComponent, IdComponent, IdTCPConnection,
  IdTCPClient;
  {$ENDIF}

type
  TMainForm = class(TForm)
    PButtons: TPanel;
    BSave: TButton;
    LLBDirs: TLabel;
    LLState: TLabel;
    LStatus: TLabel;
    BStopStart: TButton;
    ActionList1: TActionList;
    ASave: TAction;
    AStopStart: TAction;
    BPauseContinue: TButton;
    APauseContinue: TAction;
    BDirAdd: TButton;
    BDirEdit: TButton;
    BDirDelete: TButton;
    ABackupAdd: TAction;
    ABackupEdit: TAction;
    ABackupDelete: TAction;
    SGBackups: TStringGrid;
    GroupBox1: TGroupBox;
    LEtime: TLabel;
    LESize: TLabel;
    LEExtensions: TLabel;
    METime: TMaskEdit;
    EBackupDir: TEdit;
    EExtension: TEdit;
    SBBackupDir: TSpeedButton;
    LELogDir: TLabel;
    ELogDir: TEdit;
    SBlogDir: TSpeedButton;
    BDefaults: TButton;
    ADefaults: TAction;
    Button1: TButton;
    AServer: TAction;
    LLServer: TLabel;
    LServer: TLabel;
    idClient: TIdTCPClient;
    procedure FormShow(Sender: TObject);
    procedure ASaveExecute(Sender: TObject);
    procedure ASaveUpdate(Sender: TObject);
    procedure AStopStartUpdate(Sender: TObject);
    procedure AStopStartExecute(Sender: TObject);
    procedure APauseContinueUpdate(Sender: TObject);
    procedure APauseContinueExecute(Sender: TObject);
    procedure ABackupAddExecute(Sender: TObject);
    procedure ABackupEditExecute(Sender: TObject);
    procedure HaveDir(Sender: TObject);
    procedure ABackupDeleteExecute(Sender: TObject);
    procedure SBBackupDirClick(Sender: TObject);
    procedure SBlogDirClick(Sender: TObject);
    procedure ADefaultsExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure AServerExecute(Sender: TObject);
    procedure SGBackupsDblClick(Sender: TObject);
  private
    FRemoteServer : String;
    fRemotePort   : Integer;
    FlastSTatus : TBackupServiceStatus;
    procedure GetConfig;
    procedure GetBackupList;
    procedure GetServiceStatus;
    procedure SaveConfig;
    procedure SetRowData(Index: Integer; EntryName: String);
    procedure SelectDir(E: TCustomEdit);
    procedure SetNotModified;
    function GetRemote: Boolean;
    function DoDeleteEntry(Entry: String): Boolean;
    function DoEnumEntries(List: TStrings): Boolean;
    function DoReadConfig(Entry: String;
      var Config: TBackupConfig): Boolean; overload;
    function DoReadDefaults(var Config: TBackupDefaults): boolean;
    function DoWriteConfig(Entry: String;
      const Config: TBackupConfig): Boolean; overload;
    function DoWriteDefaults(var Config: TBackupDefaults): Boolean;
    procedure DoContinueBackupService;
    function DoGetBackupServiceStatus: TBackupServiceStatus;
    procedure DoPauseBackupService;
    procedure DoReloadBackupService;
    procedure DoStartBackupService;
    procedure DoStopBackupService;
    procedure InitForm;
    { Private declarations }
  public
    { Public declarations }
    Property RemoteConfig : Boolean Read GetRemote;
  end;

var
  MainForm: TMainForm;

implementation

{$IFDEF WIN32}
  {$R *.dfm}
{$ELSE}
  {$R *.xfm}
{$ENDIF}

uses FrmBackupEntry,{$IFDEF WIN32}filectrl,{$ENDIF}ctrlBackup, remotefibssclient, frmServer;


ResourceString
  SStop       = 'St&op';
  SSTart      = 'St&art';
  SPause      = '&Pause';
  SContinue   = '&Continue';
  SName       = 'Name';
  SServer     = 'Server';
  SDatabase   = 'Database path';
  SBackupFile = 'backup file';
  SExtensions = 'Extensions';
  SConfirmDelete = 'Really delete directory "%s"?';
  SNotInstalled = 'Not installed';
  SKeyNotDetermined = 'The backup scheduling service is not installed.'#13#10+
                      'Can not determine what registry key to use, defaulting to HKEY_LOCAL_MACHINE';
  SMustRunAs = 'The backup service is configured to run as user %s.'#13#10+
               'To configure the service, this program must be run by that user.';
  SErrReadingConfig = 'Could not read configuration data';
  SErrWritingConfig = 'Could not save configuration data';
  SLocalMachine = 'Local machine';
  SRemoteServer = 'Remote: %s  Version: %d';

Procedure TMainForm.SaveConfig;

Var
  ReReadConfig : Boolean;
  Config : TBackupDefaults;

begin
  ReReadConfig:=METime.Modified;
  With Config do
    begin
    BackupTime:=StrToTime(METime.Text);
    BackupDir:=EBackupDir.Text;
    LogDir:=ELogDir.Text;
    BackupExt:=EExtension.Text;
    end;
  if Not DoWriteDefaults(Config) then
    MessageDLG(SErrWritingConfig,mtError,[MBOK],0)
  else
    SetNotModified;
  If ReReadConfig and (FLastStatus=bssRunning) then
    DoReloadBackupService;
end;

procedure TMainForm.SetNotModified;

begin
  METime.Modified:=False;
  EBackupDir.Modified:=False;
  ELogDir.Modified:=False;
  EExtension.Modified:=False;
end;

procedure TMainForm.FormShow(Sender: TObject);
begin
  SGBackups.Cells[0,0]:=SName;
  SGBackups.Cells[1,0]:=SServer;
  SGBackups.Cells[2,0]:=SDatabase;
  SGBackups.Cells[3,0]:=SBackupFile;
  InitForm;
end;

procedure TMainForm.InitForm;
begin
  if idClient.Connected then
    idClient.Disconnect;
    
  if (FRemoteServer='') then
  begin
    LServer.Caption:=SLocalMachine;
  end
  else
  begin
    idClient.Port:=fRemotePort;
    idClient.Host:=fRemoteServer;
    try
      idClient.Connect;
    except
      on E:Exception do
      begin
        ShowMessage(E.Message);
        fRemoteServer:='';
        Exit;
      end;
    end;
    ProcessHandshake(idClient);
    LServer.Caption:=Format(SRemoteServer,[RemoteServerName,RemoteServerVersion]);
  end;
  GetServiceStatus;
  GetConfig;
  GetBackupList;
end;

procedure TMainForm.GetConfig;

Var
  Config : TBackupDefaults;

begin
  if DoReadDefaults(Config) then
    With Config do
      begin
      METime.Text:=FormatDateTime('hh:nn',BackupTime);
      EBackupDir.Text:=BackupDir;
      ELogDir.Text:=LogDir;
      EExtension.Text:=BackupExt;
      end
   else
     MessageDLG(SErrReadingConfig,mtError,[mbOK],0);
end;

procedure TMainForm.GetServiceStatus;

begin
  FLastStatus:=DoGetBackupServiceStatus;
  If FLastStatus=bssRunning then
    LStatus.Font.Color:=clGreen
  else
    LStatus.Font.Color:=clRed;
  LStatus.Caption:=GetStatusDescription(FLastStatus);
  If (FLastStatus=bssStopped) then
    AStopStart.Caption:=SStart
  else
    AStopStart.Caption:=SStop;
  if FLastStatus=bsspaused then
    APauseContinue.Caption:=SContinue
  else
    APauseContinue.Caption:=SPause;
end;

procedure TMainForm.SetRowData(Index : Integer; EntryName : String);

Var
 Config : TBackupConfig;

begin
  SGBackups.Cells[0,Index]:=EntryName;
  If DoReadConfig(EntryName,Config) then
    begin
    SGBackups.Cells[1,Index]:='';
    SGBackups.Cells[2,Index]:='';
    SGBackups.Cells[3,Index]:='';
    With SGBackups,Config do
      begin
      SGBackups.Cells[1,Index]:=ServerName;
      SGBackups.Cells[2,Index]:=DatabasePath;
      SGBackups.Cells[3,Index]:=BackupFile;
      end;
    end;
end;

procedure TMainForm.GetBackupList;

Var
  I : Integer;
  L : TStrings;

begin
  L:=TStringList.Create;
  Try
    //Clear grid
    SGBackups.RowCount:=2;
    for i := 0 to 3 do
      SGBackups.Cells[i,1]:='';

    DoEnumEntries(L);
    If L.Count=0 then
      SGBackups.RowCount:=2
    else
      SGBackups.RowCount:=L.Count+1;
    For I:=1 to L.Count do
      SetRowData(I,L[i-1]);
  Finally
    L.Free;
  end;
end;

procedure TMainForm.ASaveExecute(Sender: TObject);
begin
  SaveConfig;
end;

procedure TMainForm.ASaveUpdate(Sender: TObject);
begin
  (Sender as TAction).Enabled:=METime.Modified or
                               EBackupDir.Modified or
                               ELogDir.Modified or
                               EExtension.Modified;
end;

procedure TMainForm.AStopStartUpdate(Sender: TObject);
begin
  (Sender as TAction).Enabled:=FLastStatus in [bssRunning,bssStopped];
end;

procedure TMainForm.AStopStartExecute(Sender: TObject);
begin
  If FLastStatus=bssStopped then
    DoStartBackupService
  else
    DoStopBackupService;
  Sleep(500); // Give time to stop
  GetServiceStatus;
end;

procedure TMainForm.APauseContinueUpdate(Sender: TObject);
begin
  (Sender as TAction).Enabled:=FLastStatus in [bssRunning,bssPaused];
end;

procedure TMainForm.APauseContinueExecute(Sender: TObject);
begin
  If (FLastStatus=bssPaused) then
    DoContinueBackupService
  else
    DoPauseBackupService;
  Sleep(500); // Give time to stop
  GetServiceStatus;
end;

procedure TMainForm.ABackupAddExecute(Sender: TObject);
begin
  With TBackupEntryForm.Create(Self) do
    try
      If RemoteConfig then
        idClient:=Self.idClient;
      If ShowModal=mrOK then
        With SGBackups do
          begin
          if (Cells[0,RowCount-1]<>'') then
            RowCount:=RowCount+1;
          SetRowData(RowCount-1,EName.Text);
          end;
    finally
      Free;
    end;
end;

procedure TMainForm.ABackupEditExecute(Sender: TObject);

Var
  N : String;

begin
  With TBackupEntryForm.Create(Self) do
    try
      If RemoteConfig then
        idClient:=Self.idClient;
      With SGBackups do
        N:=Cells[0,Row];
      EName.Text:=N;
      Ename.Enabled:=False;
      If ShowModal=mrOK then
        SetRowData(SGBackups.Row,N);
    finally
      Free;
    end;
end;

procedure TMainForm.HaveDir(Sender: TObject);
begin
  With SGBackups do
    (Sender as Taction).Enabled:=(Row>0) and (Cells[0,Row]<>'');
end;

procedure TMainForm.ABackupDeleteExecute(Sender: TObject);

Var
  DN : String;
  i: Integer;
begin
  With SGBackups do
    DN:=Cells[0,Row];
  If MessageDLG(Format(SConfirmDelete,[DN]),mtConfirmation,[mbYes,mbNo],0)=mrYes then
    if DoDeleteEntry(DN) then
    begin
      with SGBackups do
      begin
        for i := Row to RowCount - 2 do    // Iterate
        begin
          SGBackups.Cells[0,i]:=SGBackups.Cells[0,i+1];
          SGBackups.Cells[1,i]:=SGBackups.Cells[1,i+1];
          SGBackups.Cells[2,i]:=SGBackups.Cells[2,i+1];
          SGBackups.Cells[3,i]:=SGBackups.Cells[3,i+1];
        end;    // for
        RowCount:=RowCount-1;
      end;    // with
    end;

end;

procedure TMainForm.SBBackupDirClick(Sender: TObject);

begin
  SelectDir(EBackupDir);
end;

procedure TMainForm.SBlogDirClick(Sender: TObject);
begin
  SelectDir(ELogDir);
end;

{$ifdef linux}
{$ifdef ver140}
{$define USEWIDESTRING}
{$endif}
{$endif}


procedure TMainForm.SelectDir(E : TCustomEdit);

Var
{$ifdef USEWIDESTRING}
  D : WideString;
{$else}
  D : String;
{$endif}
begin
  D:=E.Text;
  If SelectDirectory('Select Directory','',D) then
    begin
    E.Text:=D;
    E.Modified:=True;
    end;
end;

procedure TMainForm.ADefaultsExecute(Sender: TObject);

begin
  With TBackupEntryForm.Create(Self) do
    try
      If RemoteConfig then
        idClient:=Self.idClient;
      EName.Text:=SDefault;
      Ename.Enabled:=False;
      ShowModal;
    finally
      Free;
    end;
end;

Function TMainForm.DoGetBackupServiceStatus : TBackupServiceStatus;

begin
  If RemoteConfig then
    Result:=GetRemoteBackupServiceStatus(idClient)
  else
    Result:=GetBackupServiceStatus;
end;

Procedure TMainForm.DoStartBackupService;

begin
  If RemoteConfig then
    StartRemoteBackupService(idClient)
  else
    StartBackupService;
end;

Procedure TMainForm.DoStopBackupService;

begin
  If RemoteConfig then
    StopRemoteBackupService(idClient)
  else
    StopBackupService;
end;

Procedure TMainForm.DoPauseBackupService;

begin
  If RemoteConfig then
    PauseRemoteBackupService(idClient)
  else
    PauseBackupService;
end;

Procedure TMainForm.DoContinueBackupService;
begin
  If RemoteConfig then
    ContinueRemoteBackupService(idClient)
  else
    ContinueBackupService;
end;

Procedure TMainForm.DoReloadBackupService;

begin
  If RemoteConfig then
    ReloadRemoteBackupService(idClient)
  else
    ReloadBackupService;
end;

procedure TMainForm.AServerExecute(Sender: TObject);

begin
  With TServerForm.Create(Self) do
    try
      EURL.Text:=FRemoteServer;
      RBRemote.Checked:=(FremoteServer<>'');
      EPort.Text:=IntToStr(FRemotePort);
      if ShowModal=mrOK then
        begin
        If RBRemote.Checked then
        begin
          FRemoteServer:=EURL.Text;
          try
            FRemotePort:=StrToInt(EPort.Text);
          except
            FRemotePort:=3070;
          end;    
        end
        else
          FRemoteServer:='';
        InitForm;
        end;
    finally
      Free;
    end;
end;

procedure TMainForm.FormCreate(Sender: TObject);

Var
  i : Integer;

begin
  I:=0;
  FRemoteServer:='';
  fRemotePort:=3070;
  while I<ParamCount do
    begin
    Inc(i);
    If Paramstr(i)='-r' then
      begin
      If I<ParamCount then
        begin
        Inc(i);
        FRemoteServer:=Paramstr(i)
        end
      end
    end;
end;

Function TMainForm.DoDeleteEntry(Entry : String) : Boolean;

begin
  If RemoteConfig then
    Result:=DeleteRemoteEntry(idClient,Entry)
  else
    Result:=DeleteEntry(Entry);
end;


Function TMainForm.DoEnumEntries(List : TStrings) : Boolean;

begin
  If RemoteConfig then
    Result:=EnumRemoteEntries(idClient,List)
  else
    Result:=EnumEntries(List);
end;

Function TMainForm.DoReadConfig(Entry : String; Var Config : TBackupConfig) : Boolean;

begin
  If RemoteConfig then
    Result:=ReadRemoteConfig(idClient,Entry,Config)
  else
    Result:=ReadConfig(Entry,Config);
end;

Function TMainForm.DoReadDefaults(Var Config : TBackupDefaults) : boolean;

begin
  If RemoteConfig then
    Result:=ReadRemoteDefaults(idClient,Config)
  else
    Result:=ReadDefaults(Config);
end;
Function TMainForm.DoWriteConfig(Entry : String; Const Config : TBackupConfig) : Boolean;

begin
  If RemoteConfig then
    Result:=WriteRemoteConfig(idClient,Entry,Config)
  else
    Result:=WriteConfig(Entry,Config);
end;
Function TMainForm.DoWriteDefaults(Var Config : TBackupDefaults) : Boolean;

begin
  If RemoteConfig then
    Result:=WriteRemoteDefaults(idClient,Config)
  else
    Result:=WriteDefaults(Config);
end;

function TMainForm.GetRemote: Boolean;
begin
  Result:=(FRemoteServer<>'')
end;

procedure TMainForm.SGBackupsDblClick(Sender: TObject);
begin
  With SGBackups do
    If (Row>0) and (Cells[0,Row]<>'') then
      ABackupEditExecute(Self);
end;

end.
