{ ---------------------------------------------------------------------
    $Id: svcBackup.pas,v 1.2 2004/02/12 21:56:27 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup scheduling service - Linux version

    See the file COPYING.TXT included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}
unit svcBackup;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, SvcMgr, Dialogs,
  ExtCtrls, eventlog, cfgBackup;

type
  TFIBService = class(TService)
    TBackup: TTimer;
    ELBackup: TEventLog;
    procedure ServiceExecute(Sender: TService);
    procedure ServiceStart(Sender: TService; var Started: Boolean);
    procedure TBackupTimer(Sender: TObject);
    procedure ServiceStop(Sender: TService; var Stopped: Boolean);
    procedure ServicePause(Sender: TService; var Paused: Boolean);
    procedure ServiceContinue(Sender: TService; var Continued: Boolean);
    procedure ServiceCreate(Sender: TObject);
  private
    { Private declarations }
    FStatus : TBackupServiceStatus;
    FBackupTime : TDateTime;
    FBackupThread : TThread;
    function InitTimer: Boolean;
    procedure PostError(Msg: String);
    function TicksTill(Atime: TDateTime): Integer;
    procedure StartBackupThread;
    procedure BackupThreadFinished(Sender : TObject);
    procedure PostInfo(Msg: String);
    function NextbackupTime(ATime: TDateTime): TDateTime;
  Protected
    function DoCustomControl(CtrlCode: DWord): Boolean; override;
  public
    function GetServiceController: TServiceController; override;

    { Public declarations }
  end;

var
  FIBService: TFIBService;

implementation

{$R *.DFM}
{$R fclel.res}

uses
  {DateUtils,}
  thrBackup;


ResourceString
  SErrBackupThreadRunning = 'Backups are already being made';
  SErrFailedToInitTimer = 'The scheduling timer could not be initialized.';
  SFinishedBackup = 'Finished backing up. Backed up %d databases in a total time of %s.';
  SConfigChanged = 'Got configuration change message. Re-read config';
  STimerSet = 'Initialized timer. Next timer tick scheduled in %s';

procedure ServiceController(CtrlCode: DWord); stdcall;
begin
  FIBService.Controller(CtrlCode);
end;

function TFIBService.GetServiceController: TServiceController;
begin
  Result := ServiceController;
end;

procedure TFIBService.ServiceExecute(Sender: TService);
begin
  While not Terminated do
    ServiceThread.ProcessRequests(True);
end;

procedure TFIBService.ServiceStart(Sender: TService;
  var Started: Boolean);
begin
  Started:=InitTimer;
  If Not Started then
    PostError(SErrFailedToInitTimer);
  FSTatus:=bssRunning;
end;

procedure TFIBService.PostInfo(Msg : String);

begin
  try
    Try
      ELBackup.Active:=True;
      ELBackup.Info(Msg);
    Finally
      ELBackup.Active:=False;
    end;
  except
  end;
end;


procedure TFIBService.PostError(Msg : String);

begin
  try
    Try
      ELBackup.Active:=True;
      ELBackup.Error(Msg);
    Finally
      ELBackup.Active:=False;
    end;
  except
  end;
end;

procedure TFIBService.TBackupTimer(Sender: TObject);

begin
  If Assigned(FBackupThread) then
    PostError(SErrBackupThreadRunning);
  StartBackupThread;
  // Reset timer.
  Sleep(2000); // Wait ~2 secs, so the scheduled time is definitely past.
  If Not InitTimer then
    PostError(SErrFailedToInitTimer);
end;

Function TFIBService.InitTimer : Boolean;

Var
  Config : TBackupDefaults;

begin
  TBackup.Enabled:=False;
  Result:=ReadDefaults(Config);
  If Result then
    begin
    FBackupTime:=Config.BackupTime;
    PostInfo(Format(STimerSet,[FormatDateTime('hh:nn:ss',NextBackupTime(FBackupTime)-Now)]));
    TBackup.InterVal:=TicksTill(FBackupTime);
    TBackup.Enabled:=True;
    end;
end;

Function TFIBService.NextbackupTime(ATime : TDateTime) : TDateTime;

begin
  If (Time>=ATime) then
    Result:=Date+1{tomorrow}+ATime
  else
    Result:=Date+ATime;
end;

Function TFIBService.TicksTill(Atime : TDateTime) : Integer;
{$IFNDEF LINUX}
  function MilliSecondsBetween(Date1,Date2:TDateTime):Integer;
  begin
    Result:=Trunc(Abs(TimeStampToMSecs(DateTimeTOTimeStamp(Date1))-TimeStampToMSecs(DateTimeTOTimeStamp(Date2))));
  end;
{$ENDIF}
begin
  Result:=MilliSecondsBetween(Now,NextBackupTime(Atime))
end;

procedure TFIBService.StartBackupThread;

begin
  FBackupThread:=TBackupThread.Create(True);
  FBackupThread.OnTerminate:=Self.BackupThreadFinished;
  (FBackupThread as TBackupThread).OnErrorMessage:=PostError;
  (FBackupThread as TBackupThread).OnInfoMessage:=PostInfo;
  FBackupThread.Resume;
end;

procedure TFIBService.BackupThreadFinished(Sender : TObject);

begin
  With (Sender as TBackupThread) do
    PostInfo(Format(SFinishedBackup,[FTotalFiles,FormatDateTime('hh:nn:ss',FTotalTime)]));
  FBackupThread:=Nil;
end;


procedure TFIBService.ServiceStop(Sender: TService;
  var Stopped: Boolean);
begin
  TBackup.Enabled:=False;
  If Assigned(FBackupThread) then
    With FBackupThread do
      begin
      If Suspended then
        Resume;
      Terminate;
      WaitFor;
      end;
  Stopped:=True;
  FStatus:=bssStopped;
end;

procedure TFIBService.ServicePause(Sender: TService;
  var Paused: Boolean);
begin
  TBackup.Enabled:=False;
  If Assigned(FBackupThread) then
    FBackupThread.Suspend;
  FStatus:=bssPaused;
end;

procedure TFIBService.ServiceContinue(Sender: TService;
  var Continued: Boolean);
begin
  If Assigned(FBackupThread) then
    FBackupThread.Resume
  else
    InitTimer;
  FStatus:=bssRunning;
end;

procedure TFIBService.ServiceCreate(Sender: TObject);
begin
  FStatus:=bssUnknown;
  ELBackup.RegisterMessageFile('');
end;

function TFIBService.DoCustomControl(CtrlCode: DWord): Boolean;
begin
  Result:=(CtrlCode=ConfigControlCode);
  If Result and TBackup.Enabled then
    begin
    InitTimer;
    PostInfo(SConfigChanged);
    end;
end;

end.
