{ ---------------------------------------------------------------------
    $Id: svcBackup.pas,v 1.2 2003/10/13 19:48:13 michael Exp $
    Copyright (c) 2003 by Michael Van Canneyt

    Backup scheduling service - Linux version
    
    See the file COPYING.TXT included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ---------------------------------------------------------------------}
unit svcBackup;

{ Define NOGUI if an executable is required which can run without X}
{$include fibssdef.inc}


interface

uses
  LibC,SysUtils, Classes, eventlog, CfgBackup
{$ifdef NOGUI}
  ,ThreadedTimer
{$else}
  ,QTypes, QExtCtrls,QForms
{$endif}
  ;

type
  TFIBService = class(TDataModule)
    ELBackup: TEventLog;
    procedure ServiceCreate(Sender: TObject);
    procedure ServiceDestroy(Sender: TObject);
  private
    { Private declarations }
{$ifdef NOGUI}
    TBackup : TThreadedTimer;
{$else}
    TBackup : TTimer;
{$endif}
    FStatus : TBackupServiceStatus;
    FTerminated : Boolean;
    FBackupTime : TDateTime;
    FBackupThread : TThread;
    function InitTimer: Boolean;
    procedure PostError(Msg: String);
    function TicksTill(Atime: TDateTime): Integer;
    procedure BackupThreadFinished(Sender : TObject);
    procedure PostInfo(Msg: String);
    function NextbackupTime(ATime: TDateTime): TDateTime;
    procedure SetupSignals;
    procedure SetupPIDFile;
    procedure DeletePIDFile;
    procedure TBackupTimer(Sender: TObject);
  Public
    procedure StartBackup;
    procedure ServiceExecute;
    procedure ServiceStop;
    procedure ServicePause;
    procedure ServiceContinue;
    Procedure ServiceReload;
  public
    { Public declarations }
    Property Terminated : Boolean Read FTerminated;
  end;

var
  FIBService: TFIBService;

implementation

{$R *.dfm}
{$R fclel.res}

uses
  DateUtils,
  thrBackup;

ResourceString
  SErrBackupThreadRunning = 'Backups are already being made';
  SErrFailedToInitTimer = 'The scheduling timer could not be initialized.';
  SFinishedBackup = 'Finished backing up. Backed up %d databases in a total time of %s.';
  SConfigChanged = 'Got configuration change message. Re-read config';
  STimerSet = 'Initialized timer. Next timer tick scheduled in %s';
  SStopReceived = 'Received TERM signal, shutting down.';
  SErrCreatingPIDFile = 'Error creating the PID file "%s": %s';
  SErrDeletingPIDFile = 'Error deleting the PID file "%s": %s';

Const
  SleepInterval = 60*1000; // One minute interval between synchronize checks.

{ ---------------------------------------------------------------------
  Signals go through here
  ---------------------------------------------------------------------}


procedure ServiceController(Sig : Integer); cdecl;
begin
  Case Sig of
    SIGHUP  : FIBService.ServiceReload;
    SIGCONT : FIBService.ServiceContinue;
    SIGTERM : FIBService.ServiceStop;
  end;
end;

{ ---------------------------------------------------------------------
  Event handlers
  ---------------------------------------------------------------------}

procedure TFIBService.ServiceCreate(Sender: TObject);


begin
  ELBackup.RegisterMessageFile('');
{$ifdef NOGUI}
  TBackup:=TThreadedTimer.Create(Self);
{$else}
  TBackup:=TTimer.Create(Self);
{$endif}
  TBackup.Enabled:=False;
  TBackup.OnTimer:=TBackupTimer;
  SetupSignals;
  SetupPidFile;
end;

procedure TFIBService.TBackupTimer(Sender: TObject);
begin
   StartBackup;
end;

procedure TFIBService.ServiceDestroy(Sender: TObject);
begin
  DeletePIDfile;
end;

{ ---------------------------------------------------------------------
  Initialization & setup routines
  ---------------------------------------------------------------------}

Procedure TFIBService.SetupSignals;

Var
  Action : TSigAction;

begin
  Action.__Sigaction_handler:=ServiceController;
  sigemptyset(Action.sa_mask);
  Action.sa_flags:=0;
  sigaction(SIGHUP,@Action,Nil);
  sigaction(SIGCONT,@Action,Nil);
  sigaction(SIGTERM,@Action,Nil);
end;

Procedure TFIBService.SetupPIDFile;

Var
  FN,S : String;

begin
  Try
    FN:=GetPIDFileName;
    S:=IntToStr(GetPID);
    With TFileStream.Create(FN,fmCreate) do
      try
        Write(S[1],Length(S));
      finally
        Free;
      end;
  except
    On E : Exception do
      PostError(Format(SErrCreatingPIDFile,[FN,E.Message]));
  end;
end;

Procedure TFIBService.DeletePIDFile;

Var
  FN : String;

begin
  Try
    FN:=GetPIDFileName;
    If Not DeleteFile(FN) then
      RaiseLastOSError;
  except
    On E : Exception do
      PostError(Format(SErrDeletingPIDFile,[FN,E.Message]));
  end;
end;

{ ---------------------------------------------------------------------
  Auxiliary routines
  ---------------------------------------------------------------------}

procedure TFIBService.PostInfo(Msg : String);

begin
  try
    Try
      ELBackup.Active:=True;
      ELBackup.Info(Msg);
    Finally
      ELBackup.Active:=False;
    end;
  except
  end;
end;

procedure TFIBService.PostError(Msg : String);

begin
  try
    Try
      ELBackup.Active:=True;
      ELBackup.Error(Msg);
    Finally
      ELBackup.Active:=False;
    end;
  except
  end;
end;

Function TFIBService.NextbackupTime(ATime : TDateTime) : TDateTime;

begin
  If (Time>=ATime) then
    Result:=Tomorrow+ATime
  else
    Result:=Today+ATime;
end;


Function TFIBService.TicksTill(Atime : TDateTime) : Integer;

begin
  Result:=MilliSecondsBetween(Now,NextBackupTime(Atime))
end;


{ ---------------------------------------------------------------------
  Routines that do the real work.
  ---------------------------------------------------------------------}

procedure TFIBService.ServiceExecute;

begin
  FStatus:=bssRunning;
  FTerminated:=False;
  if not InitTimer then
    PostError(SErrFailedToInitTimer);
  While Not Terminated do
    begin
{$ifdef NOGUI}
    Sleep(SleepInterval);
    CheckSynchronize;
{$else}
     // _Not_ processMessages, this takes 100% CPU time.
     Application.HandleMessage;
{$endif}
    If (FStatus=bssReload) then
      begin
      FStatus:=bssRunning;
      if not InitTimer then
        PostError(SErrFailedToInitTimer);
      end;
    end;
  TBackup.Enabled:=False;
  If (FStatus=bssStopped) then
    PostINfo(SStopReceived);
end;

Function TFIBService.InitTimer : Boolean;

Var
  Config : TBackupDefaults;

begin
  Result:=ReadDefaults(Config);
  If Result then
    begin
    FBackupTime:=Config.BackupTime;
    PostInfo(Format(STimerSet,[FormatDateTime('hh:nn:ss',NextBackupTime(FBackupTime)-Now)]));
    TBackup.Interval:=TicksTill(FbackupTime);
    TBackup.Enabled:=True;
    end;
end;


procedure TFIBService.StartBackup;

begin
  TBackup.Enabled:=False;
  If Assigned(FBackupThread) then
    PostError(SErrBackupThreadRunning);
  FBackupThread:=TBackupThread.Create(True);
  FBackupThread.FreeOnTerminate:=True;
  FBackupThread.OnTerminate:=Self.BackupThreadFinished;
  (FBackupThread as TBackupThread).OnErrorMessage:=PostError;
  (FBackupThread as TBackupThread).OnInfoMessage:=PostInfo;
  FBackupThread.Resume;
  InitTimer;
end;

procedure TFIBService.BackupThreadFinished(Sender : TObject);

begin
  With (Sender as TBackupThread) do
    PostInfo(Format(SFinishedBackup,[FTotalFiles,FormatDateTime('hh:nn:ss',FTotalTime)]));
  FBackupThread:=Nil;
end;

{ ---------------------------------------------------------------------
  Called when a signal arrives.
  ---------------------------------------------------------------------}

procedure TFIBService.ServiceStop;

begin
  If Assigned(FBackupThread) then
    With FBackupThread do
      begin
      If Suspended then
        Resume;
      Terminate;
      WaitFor;
      end;
  FStatus:=bssStopped;
  FTerminated:=True;
end;

procedure TFIBService.ServicePause;
begin
  If Assigned(FBackupThread) then
    FBackupThread.Suspend;
  FStatus:=bssPaused;
end;

procedure TFIBService.ServiceContinue;
begin
  If Assigned(FBackupThread) then
    FBackupThread.Resume;
  FStatus:=bssRunning;
end;

Procedure TFIBService.ServiceReload;
begin
  FStatus:=bssReload;
  PostInfo(SConfigChanged);
end;


end.
