program simpleservice;

{ $APPTYPE CONSOLE}

uses
  Windows,SysUtils,winsvc;

Const
  PSimpleService : PChar = 'SimpleService';

resourcestring
  SErrUnknownCode = 'Unknown control opcode : %d';
  SErrRegistering = 'Error registering services : %s';
  SErrControlPort = 'Error creating control port : %s';

Procedure ServiceError(Fmt : String; Args : Array of const);

begin
  OutputDebugString(PChar(Format(Fmt,Args)));
end;

Procedure ReportOSError(Msg : String);

begin
  try
    RaiseLastWin32Error;// This will format the error for us.
  except
    On E : Exception do
      ServiceError(Msg,[E.Message]);
  end;
end;

Const
  cmdPipe    = 1;  // Someone accesses the named pipe.
  cmdControl = 2;  // Control handler received a control command.

Var
  CurrentStatus : TServiceStatus;
  ControlPort : THandle;
  ServiceStatusHandle : THandle;
  HPipe : Thandle = 0;
  RunAsService : Boolean;
  PO : POverlapped;
  O : Overlapped;

Procedure ConnectPipe;

begin
  FillChar(o, sizeof(o),0);
  ConnectNamedPipe(hpipe, @o);
end;

Procedure CreatePipe;

begin
  // Create pipe
  hPipe:=CreateNamedPipe('\\.\pipe\SimpleService', PIPE_ACCESS_OUTBOUND or FILE_FLAG_OVERLAPPED,
                         PIPE_TYPE_BYTE, 1, SizeOf(TDateTime), SizeOf(TDateTime),
                         1000, Nil);
  // Associate the pipe with the control port
  CreateIoCompletionPort(HPipe,ControlPort,cmdPipe,0);
  // Pend an asynchronous connect against the pipe
  ConnectPipe;
end;

Procedure ClosePipe;

begin
  If (HPipe<>0) then
    begin
    CloseHandle(HPipe);
    HPipe:=0;
    end;
end;

Procedure HandlePipe;

Var
  D                : TDateTime;
  BytesTransferred : DWord;

begin
  D:=Now;
  WriteFile(hpipe,D,SizeOf(D), BytesTransferred, Nil);
  FlushFileBuffers(hpipe);
  DisconnectNamedPipe(hpipe);
  ConnectPipe;
end;

Procedure SimpleServiceCtrlHandler (Opcode : DWord); StdCall;

begin
  Case Opcode of
    SERVICE_CONTROL_PAUSE :
      begin
      ClosePipe;
      CurrentStatus.dwCurrentState:=SERVICE_PAUSED;
      end;
    SERVICE_CONTROL_STOP :
      begin
      ClosePipe;
      CurrentStatus.dwCurrentState:=SERVICE_STOPPED;
      end;
    SERVICE_CONTROL_CONTINUE :
      begin
      CreatePipe;
      CurrentStatus.dwCurrentState:=SERVICE_RUNNING;
      end;
    SERVICE_CONTROL_INTERROGATE : ;
  else
    ServiceError(SErrUnknownCode,[Opcode]);
  end;
  SetServiceStatus(ServiceStatusHandle,CurrentStatus);
  // Notify main thread that control code came in, so it can take action.
  PostQueuedCompletionStatus(ControlPort,0,cmdControl,NiL);
end;

{$ifdef ver130}
Type
  PPChar = ^PChar;
{$endif}  

Procedure SimpleServiceMain (Argc : DWord; Argsv :PPChar);stdcall;

Var
  BytesTransferred : DWord;
  Command : Dword;

begin
  ControlPort:=CreateIoCompletionPort(INVALID_HANDLE_VALUE, 0, cmdPipe, 0);
  If (ControlPort=0) then
    ReportOSError(SErrControlPort);
  If RunAsService then
    begin
    // Initialize status record.
    FillChar(CurrentStatus,SizeOf(CurrentStatus),0);
    With CurrentStatus do
      begin
      dwServiceType:=SERVICE_WIN32_OWN_PROCESS;
      dwCurrentState:=SERVICE_START_PENDING;
      dwControlsAccepted:=SERVICE_ACCEPT_STOP or
                          SERVICE_ACCEPT_PAUSE_CONTINUE;
      end;
    ServiceStatusHandle:=RegisterServiceCtrlHandler('SimpleService',@SimpleServiceCtrlHandler);
    SetServiceStatus(ServiceStatusHandle,CurrentStatus);
    end;
  CreatePipe;
  If RunAsService then
    begin
    CurrentStatus.dwCurrentState:=SERVICE_RUNNING;
    SetServiceStatus(ServiceStatusHandle,CurrentStatus);
    end;
  PO:=Nil;
  Repeat
    // Wait for either control code notification or client connect
    GetQueuedCompletionStatus(ControlPort,BytesTransferred,Command,po,INFINITE);
    If Command=cmdPipe then
      HandlePipe;
     // otherwise a Control code received, do nothing, wait for new command
  Until RunAsService and (CurrentStatus.dwCurrentState=SERVICE_STOPPED);
end;

Var
  ServiceTable : Array[0..1] of TServiceTableEntry = (
   (lpServiceName: 'SimpleService';lpServiceProc:@SimpleServiceMain),
   (lpServiceName: Nil;lpServiceProc:Nil)
   );

Procedure RegisterServiceLoop;

begin
  If not StartServiceCtrlDispatcher(ServiceTable[0]) then
    ReportOSError(SErrRegistering);
end;

begin
  RunAsService:=(ParamCount=1) and (Paramstr(1)='/RUN');
  If RunAsService then
    RegisterServiceLoop
  else
    SimpleServiceMain(0,Nil);
end.
